(function(){
  const qs = (s) => document.querySelector(s);

  const inputEl   = qs('#inputText');
  const outputEl  = qs('#outputText');
  const copyBtn   = qs('#copyBtn');
  const swapBtn   = qs('#swapBtn');
  const transBtn  = qs('#translateBtn');
  const toast     = qs('#toast');
  const clearBtn  = qs('#clearInputBtn');

  const dirIdEn = qs('#id-en');
  const dirEnId = qs('#en-id');
  const srcBadge = qs('#srcBadge');
  const tgtBadge = qs('#tgtBadge');
  const charCounter = qs('#charCounter');

  let lastController = null; // untuk cancel request sebelumnya

  function getLangPair(){
    return dirIdEn.checked ? ['id', 'en'] : ['en', 'id'];
  }

  function setBadges(){
    const [src, tgt] = getLangPair();
    srcBadge.textContent = src.toUpperCase();
    tgtBadge.textContent = tgt.toUpperCase();
    inputEl.placeholder = src === 'id'
      ? 'Ketik atau tempel teks Bahasa Indonesia di sini...'
      : 'Type or paste English text here...';
  }

  function showToast(msg, type){
    toast.textContent = msg;
    toast.classList.remove('error', 'show');
    if(type === 'error') toast.classList.add('error');
    void toast.offsetWidth; // restart state
    toast.classList.add('show');
    setTimeout(() => toast.classList.remove('show'), 2000);
  }

  function setLoading(isLoading){
    if(isLoading){
      transBtn.classList.add('loading');
      transBtn.setAttribute('aria-busy', 'true');
      transBtn.disabled = true;
    } else {
      transBtn.classList.remove('loading');
      transBtn.removeAttribute('aria-busy');
      transBtn.disabled = false;
    }
  }

  function updateCounter(){
    const len = (inputEl.value || '').length;
    charCounter.textContent = len.toString();
  }

  async function doTranslate(){
    const text = (inputEl.value || '').trim();
    if(!text){
      showToast('Teks tidak boleh kosong.', 'error');
      inputEl.focus();
      return;
    }
    if(!navigator.onLine){
      showToast('Perangkat offline. Cek koneksi internet.', 'error');
      return;
    }

    const [source, target] = getLangPair();

    // Batalkan request lama (kalau user klik berkali-kali)
    if (lastController) { try { lastController.abort(); } catch(e){} }
    const controller = new AbortController();
    lastController = controller;

    setLoading(true);
    try{
      const res = await fetch('api/translate', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ text, source, target }),
        signal: controller.signal
      });

      const data = await res.json().catch(() => ({}));

      if(!res.ok || !data.success){
        // jika aborted, jangan tampilkan error
        if (controller.signal.aborted) return;
        const msg = (data && data.error) ? data.error : ('Terjadi kesalahan (' + res.status + ').');
        throw new Error(msg);
      }

      outputEl.textContent = data.translation || '';
      showToast('Berhasil diterjemahkan.');
    } catch (err){
      if (controller.signal.aborted) return; // abaikan error dari abort
      showToast(err.message || 'Gagal menerjemahkan.', 'error');
    } finally{
      setLoading(false);
    }
  }

  function swapDirection(){
    const wasIdEn = dirIdEn.checked;
    if(wasIdEn){ dirEnId.checked = true; } else { dirIdEn.checked = true; }
    // Tukar isi agar praktis
    const inputVal = inputEl.value;
    const outputVal = outputEl.textContent;
    inputEl.value = outputVal;
    outputEl.textContent = inputVal;
    updateCounter();
    setBadges();
  }

  async function copyOutput(){
    const text = outputEl.textContent || '';
    if(!text){ showToast('Tidak ada teks untuk disalin.', 'error'); return; }
    try{
      await navigator.clipboard.writeText(text);
      showToast('Hasil disalin.');
    } catch(e){
      showToast('Gagal menyalin ke clipboard.', 'error');
    }
  }

  function clearInput(){
    inputEl.value = '';
    updateCounter();
    inputEl.focus();
  }

  // Events
  transBtn.addEventListener('click', doTranslate);
  swapBtn.addEventListener('click', swapDirection);
  copyBtn.addEventListener('click', copyOutput);
  clearBtn.addEventListener('click', clearInput);

  inputEl.addEventListener('input', updateCounter);
  inputEl.addEventListener('keydown', (e) => {
    if(e.ctrlKey && e.key === 'Enter'){
      e.preventDefault();
      doTranslate();
    }
  });

  dirIdEn.addEventListener('change', setBadges);
  dirEnId.addEventListener('change', setBadges);

  // Init
  setBadges();
  updateCounter();
})();
